<?php
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__.'/error.log');
error_reporting(E_ALL);

session_start();

error_log("This is a test error log");

include '../config_filepath.php';
include '../database/db_config.php';
include '../database/api_config.php';

$rawInput = file_get_contents("php://input");
$data = json_decode($rawInput, true);

$communityID = isset($data['community_id']) ? trim($data['community_id']) : null;
$members = isset($data['members']) && is_array($data['members']) ? $data['members'] : [];

if (!$communityID || !preg_match('/^[\d-]{10,31}@g\.us$/', $communityID)) {
    echo json_encode(['success' => false, 'message' => 'Invalid or missing community ID']);
    exit;
}

if (empty($members)) {
    echo json_encode(['success' => false, 'message' => 'No members selected']);
    exit;
}

$waIDs = [];
foreach ($members as $m) {
    $empid = $m['empid'] ?? '';
    $phoneRaw = $m['phone'] ?? '';
    $cleanPhone = preg_replace('/\D/', '', $phoneRaw);

    if ($cleanPhone && strlen($cleanPhone) >= 10 && strlen($cleanPhone) <= 15) {
        $waIDs[$cleanPhone] = $empid;
    }
}

if (empty($waIDs)) {
    echo json_encode(['success' => false, 'message' => 'No valid phone numbers found']);
    exit;
}

$token = API_TOKEN;
$communityUrl = "https://gate.whapi.cloud/communities/" . urlencode($communityID);

$ch = curl_init($communityUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . $token,
        'Accept: application/json'
    ],
    CURLOPT_SSL_VERIFYPEER => false
]);
$response = curl_exec($ch);
curl_close($ch);

$communityData = json_decode($response, true);

if (!isset($communityData['invite_code'], $communityData['participants'])) {
    echo json_encode(['success' => false, 'message' => 'Failed to get community info']);
    exit;
}

$inviteCode = $communityData['invite_code'];
$existingMembers = array_map(function($p) {
    return preg_replace('/\D/', '', $p['id']);
}, $communityData['participants']);

$added = [];
$invited = [];
$failed = [];

foreach (array_keys($waIDs) as $phone) {
    if (in_array($phone, $existingMembers)) {
        $added[] = $phone . ' (already a member)';
        continue;
    }

    $addResult = addToCommunity($communityID, $phone, $token);

    if ($addResult['success']) {
        $added[] = $phone;
    } else {
        $inviteResult = sendCommunityInvite($inviteCode, $phone, $token);
        if ($inviteResult['success']) {
            $invited[] = $phone;
        } else {
            $failed[] = $phone;
        }
    }
}

echo json_encode([
    'success' => true,
    'message' => "Added: " . count($added) . ", Invited: " . count($invited) . ", Failed: " . count($failed),
    'added' => $added,
    'invited' => $invited,
    'failed' => $failed
]);
function addToCommunity($communityID, $phone, $token) {
    $url = "https://gate.whapi.cloud/communities/" . urlencode($communityID) . "/participants";
    $payload = json_encode(['participants' => [$phone]]);
    $headers = [
        'Authorization: Bearer ' . $token,
        'Accept: application/json',
        'Content-Type: application/json'
    ];

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $result = json_decode($response, true);

    return [
        'success' => $httpCode === 200 && isset($result['success']) && $result['success'],
        'response' => $result
    ];
}

function sendCommunityInvite($inviteCode, $phone, $token) {
    $url = "https://gate.whapi.cloud/groups/link/" . $inviteCode;
    $payload = json_encode([
        "to" => (string)$phone,
        "preview_type" => "none",
        "body" => "Hi! Please reply to this message to activate your community invitation link: %URL%"
    ]);
    $headers = [
        "Accept: application/json",
        "Authorization: Bearer " . $token,
        "Content-Type: application/json"
    ];

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false,
    ]);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    error_log("Invite API response for phone $phone: HTTP $httpCode - $response");
    $data = json_decode($response, true);

    return [
        'success' => $httpCode === 200 && !isset($data['error']),
        'response' => $data
    ];
}
